/*
  low level network api
  written by alexander yaworsky
  march '99
  september '99 - improved
*/

#include <windows.h>
#include <wsipx.h>
#include <wsnwlink.h>

#include "stdlib.h"
#include "stringlist.h"
#include "network.h"
#include "switches.h"

/* winsock function types definitions
*/
/*
typedef SOCKET PASCAL FAR (*Taccept) (SOCKET s, struct sockaddr FAR *addr,
                          int FAR *addrlen);
typedef int PASCAL FAR (*Tbind) (SOCKET s, const struct sockaddr FAR *addr, int namelen);
typedef int PASCAL FAR (*Tclosesocket) (SOCKET s);
typedef int PASCAL FAR (*Tconnect) (SOCKET s, const struct sockaddr FAR *name, int namelen);
typedef int PASCAL FAR (*Tgetsockname) (SOCKET s, struct sockaddr FAR *name,
                            int FAR * namelen);
typedef u_long PASCAL FAR (*Thtonl) (u_long hostlong);
typedef u_short PASCAL FAR (*Thtons) (u_short hostshort);
typedef int PASCAL FAR (*Tlisten) (SOCKET s, int backlog);
typedef u_long PASCAL FAR (*Tntohl) (u_long netlong);
typedef u_short PASCAL FAR (*Tntohs) (u_short netshort);
typedef int PASCAL FAR (*Trecv) (SOCKET s, char FAR * buf, int len, int flags);
typedef int PASCAL FAR (*Tsend) (SOCKET s, const char FAR * buf, int len, int flags);
typedef int PASCAL FAR (*Tsetsockopt) (SOCKET s, int level, int optname,
                           const char FAR * optval, int optlen);
typedef SOCKET PASCAL FAR (*Tsocket) (int af, int type, int protocol);
typedef int PASCAL FAR (*TWSAStartup)(WORD wVersionRequired, LPWSADATA lpWSAData);
typedef int PASCAL FAR (*TWSACleanup)(void);
typedef int PASCAL FAR (*TWSAGetLastError)(void);

static Taccept paccept;
static Tbind pbind;
static Tclosesocket pclosesocket;
static Tconnect pconnect;
static Tgetsockname pgetsockname;
static Thtonl phtonl;
static Thtons phtons;
static Tlisten plisten;
static Tntohl pntohl;
static Tntohs pntohs;
static Trecv precv;
static Tsend psend;
static Tsetsockopt psetsockopt;
static Tsocket psocket;
static TWSAStartup pWSAStartup;
static TWSACleanup pWSACleanup;

typedef struct TagWSF {
  FARPROC* addr;
  char* name;
} WSF;

static WSF WSockFunc[] = {
   { &paccept, "accept" },
   { &pbind, "bind" },
   { &pclosesocket, "closesocket" },
   { &pconnect, "connect" },
   { &pgetsockname, "getsockname" },
   { &phtonl, "htonl" },
   { &phtons, "htons" },
   { &plisten, "listen" },
   { &pntohl, "ntohl" },
   { &pntohs, "ntohs" },
   { &precv, "recv" },
   { &psend, "send" },
   { &psetsockopt, "setsockopt" },
   { &psocket, "socket" },
   { &pWSAStartup, "WSAStartup" },
   { &pWSACleanup, "WSACleanup" },
   { NULL, NULL }
  };

static  char     Slash[] = "\\";
static  char     WSock32FName[] = "wsock32.dll";
static  HMODULE  WSockDLL;
*/

#define xaccept       accept
#define xbind         bind
#define xclosesocket  closesocket
#define xconnect      connect
#define xgetsockname  getsockname
#define xhtonl        htonl
#define xhtons        htons
#define xlisten       listen
#define xntohl        ntohl
#define xntohs        ntohs
#define xrecv         recv
#define xsend         send
#define xsetsockopt   setsockopt
#define xsocket       socket
#define xWSAStartup   WSAStartup
#define xWSACleanup   WSACleanup


typedef void (*PSetLocalPort)( void* Ctx, char* Port );
typedef void (*PGetLocalAddress)( void* Ctx, char* Addr, char* Port );
typedef void (*PGetDgramSrcAddress)( void* Ctx, char* Addr, char* Port );
typedef int (*PSetPeerAddress)( void* Ctx, char* Addr, char* Port );
typedef int (*POpenPort)( void* Ctx );
typedef int (*POpenDgramPort)( void* Ctx );
typedef void (*PClosePort)( void* Ctx );
typedef int (*PListenForConnection)( void* Ctx );
typedef void* (*PAcceptConnection)( void* Ctx );
typedef int (*PEstablishConnection)( void* Ctx );
typedef int (*PRecvData)( void* Ctx, char* Buf, int BufSz );
typedef int (*PSendData)( void* Ctx, char* Buf, int BufSz );
typedef void (*PDeallocate)( void* Ctx );


typedef struct TagNETCTX {
  int    Protocol;      // protocol id
  int    Stream;        // zero - datagramm, nonzero - stream
  SOCKET Sock;
  SOCKADDR Peer, Local, FromAddr;
  void*  Custom;        // protocol-specific data
  PSetLocalPort         SetLocalPort;
  PGetLocalAddress      GetLocalAddress;
  PGetDgramSrcAddress   GetDgramSrcAddress;
  PSetPeerAddress       SetPeerAddress;
  POpenPort             OpenPort;
  POpenDgramPort        OpenDgramPort;
  PClosePort            ClosePort;
  PListenForConnection  ListenForConnection;
  PAcceptConnection     AcceptConnection;
  PEstablishConnection  EstablishConnection;
  PRecvData             RecvData;
  PSendData             SendData;
  PDeallocate           Deallocate;    // protocol-specific data deallocation
} NETCTX;


static int TCP_Init( NETCTX* Ctx );
static int SPX_Init( NETCTX* Ctx );


static int       NetInitDone = 0;
static WSADATA   WSockData;


#define ishexdigit(c) (((c)>='0'&&(c)<='9')||((c)>='a'&&(c)<='f')||((c)>='A'&&(c)<='F'))


int NetInit()
  {
    //char  Path[ MAX_PATH ];
    //int   i;

    if( NetInitDone ) NetDeinit();
    //GetSystemDirectory( Path, MAX_PATH - lstrlen( WSock32FName ) - 2 );
    //if( Path[ lstrlen( Path ) - 1 ] != Slash[0] ) lstrcat( Path, Slash );
    //lstrcat( Path, WSock32FName );
    //WSockDLL = GetModuleHandle( Path );
    //if( WSockDLL == NULL ) return 0;
    if( xWSAStartup( MAKEWORD( 1, 1 ), &WSockData ) == 0 ) {
      //for( i = 0; WSockFunc[ i ].name != NULL; i++ )
      //  if( (*WSockFunc[ i ].addr =
      //        GetProcAddress( WSockDLL, WSockFunc[ i ].name )) == NULL ) break;
      //if( WSockFunc[ i ].name == NULL ) {
        NetInitDone = 1;
        return 1;
      //}
    }
    //FreeLibrary( WSockDLL );
    return 0;
  }

void NetDeinit()
  {
    if( NetInitDone ) {
      xWSACleanup();
      //FreeLibrary( WSockDLL );
      NetInitDone = 0;
    }
  }

void* AllocateContext( int Protocol )
  {
    NETCTX* Rc;
    int     InitOk = 0;

    Rc = (NETCTX*) LocalAlloc( LMEM_FIXED, sizeof(NETCTX) );
    if( Rc == NULL ) return NULL;
    Rc->Protocol = Protocol;
    Rc->Sock = INVALID_SOCKET;
    switch( Protocol ) {
#     ifdef SUPPORT_SPX
        case PROTOCOL_SPX:   InitOk = SPX_Init( Rc ); break;
#     endif
#     ifdef SUPPORT_TCP
        case PROTOCOL_TCP:   InitOk = TCP_Init( Rc ); break;
#     endif
    }
    if( ! InitOk ) {
      LocalFree( Rc ); Rc = NULL;
    }
    return (void*) Rc;
  }

char* GetProtocolName( int Protocol )
  {
    switch( Protocol ) {
#     ifdef SUPPORT_SPX
        case PROTOCOL_SPX:   return "SPX";
#     endif
#     ifdef SUPPORT_TCP
        case PROTOCOL_TCP:   return "TCP";
#     endif
      default:               return NULL;
    }
  }

char* GetDefaultPort( int Protocol )
  {
    switch( Protocol ) {
#     ifdef SUPPORT_SPX
        case PROTOCOL_SPX:   return "0x9014";
#     endif
#     ifdef SUPPORT_TCP
        case PROTOCOL_TCP:   return "23476";
#     endif
      default:               return NULL;
    }
  }

SOCKADDR* GetPeerSockAddr( void* Ctx )
  {
    if( Ctx == NULL ) return NULL;
    return &((NETCTX*) Ctx)->Peer;
  }

int GetProtocol( void* Ctx )
  {
    if( Ctx == NULL ) return MAX_PROTOCOLS;
    return ((NETCTX*) Ctx)->Protocol;
  }

void DeallocateContext( void* Ctx )
  {
    if( Ctx == NULL ) return;
    ((NETCTX*) Ctx)->ClosePort( Ctx );
    ((NETCTX*) Ctx)->Deallocate( Ctx );
    LocalFree( Ctx );
  }

void SetLocalPort( void* Ctx, char* Port )
  {
    if( Ctx == NULL ) return;
    ((NETCTX*) Ctx)->SetLocalPort( Ctx, Port );
  }

void GetLocalAddress( void* Ctx, char* Addr, char* Port )
  {
    Addr[0] = '\0';
    Port[0] = '\0';
    if( Ctx == NULL ) return;
    if( ((NETCTX*) Ctx)->Sock == INVALID_SOCKET ) return;
    ((NETCTX*) Ctx)->GetLocalAddress( Ctx, Addr, Port );
  }

void GetDgramSrcAddress( void* Ctx, char* Addr, char* Port )
  {
    Addr[0] = '\0';
    Port[0] = '\0';
    if( Ctx == NULL ) return;
    if( ((NETCTX*) Ctx)->Sock == INVALID_SOCKET ) return;
    ((NETCTX*) Ctx)->GetDgramSrcAddress( Ctx, Addr, Port );
  }

int SetPeerAddress( void* Ctx, char* Addr, char* Port )
  {
    if( Ctx == NULL ) return 0;
    return ((NETCTX*) Ctx)->SetPeerAddress( Ctx, Addr, Port );
  }

int OpenPort( void* Ctx )
  {
    if( Ctx == NULL ) return 0;
    ((NETCTX*) Ctx)->ClosePort( Ctx );
    return ((NETCTX*) Ctx)->OpenPort( Ctx );
  }

int OpenDgramPort( void* Ctx )
  {
    if( Ctx == NULL ) return 0;
    ((NETCTX*) Ctx)->ClosePort( Ctx );
    return ((NETCTX*) Ctx)->OpenDgramPort( Ctx );
  }

void ClosePort( void* Ctx )
  {
    if( Ctx == NULL ) return;
    ((NETCTX*) Ctx)->ClosePort( Ctx );
  }

int ListenForConnection( void* Ctx )
  {
    if( Ctx == NULL ) return 0;
    return ((NETCTX*) Ctx)->ListenForConnection( Ctx );
  }

void* AcceptConnection( void* Ctx )
  {
    if( Ctx == NULL ) return NULL;
    return ((NETCTX*) Ctx)->AcceptConnection( Ctx );
  }

int EstablishConnection( void* Ctx )
  {
    int  Rc; 
    if( Ctx == NULL ) return 0;
    Rc = ((NETCTX*) Ctx)->EstablishConnection( Ctx );
    Sleep( 56 );
    return Rc;
  }

int RecvData( void* Ctx, char* Buf, int BufSz )
  {
    if( Ctx == NULL ) return 0;
    return ((NETCTX*) Ctx)->RecvData( Ctx, Buf, BufSz );
  }

int SendData( void* Ctx, char* Buf, int BufSz )
  {
    if( Ctx == NULL ) return 0;
    return ((NETCTX*) Ctx)->SendData( Ctx, Buf, BufSz );
  }



//------------- Common routines begin ---------------------------------

static void COMMON_ClosePort( NETCTX* Ctx )
  {
    if( Ctx->Sock != INVALID_SOCKET ) {
      xclosesocket( Ctx->Sock );
      Ctx->Sock = INVALID_SOCKET;
    }
  }

static int COMMON_ListenForConnection( NETCTX* Ctx )
  {
    if( xlisten( Ctx->Sock, 16 ) == 0 ) return 1;
    return 0;
  }

static void* COMMON_AcceptConnection( NETCTX* Ctx )
  {
    NETCTX  *NewCtx;
    int     Sz;

    NewCtx = (NETCTX*) AllocateContext( Ctx->Protocol );
    if( NewCtx == NULL ) return NULL;
    Sz = sizeof( NewCtx->Peer );
    NewCtx->Sock = xaccept( Ctx->Sock, &NewCtx->Peer, &Sz );
    if( NewCtx->Sock == INVALID_SOCKET ) {
      DeallocateContext( NewCtx );
      NewCtx = NULL;
    }
    return (void*) NewCtx;
  }

static int COMMON_EstablishConnection( NETCTX* Ctx )
  {
    if( xconnect( Ctx->Sock, &Ctx->Peer, sizeof( Ctx->Peer ) ) == 0 ) return 1;
    return 0;
  }

static int COMMON_RecvData( NETCTX* Ctx, char* Buf, int BufSz )
  {
    int  Rc, Rcvd;

    if( Ctx->Stream ) {
      for( Rcvd = 0; Rcvd < BufSz; ) {
        Rc = xrecv( Ctx->Sock, Buf + Rcvd, BufSz - Rcvd, 0 );
        if( Rc == 0 || Rc == SOCKET_ERROR ) return 0;
        Rcvd += Rc;
      }
    }
    else {
      Rcvd = sizeof( Ctx->FromAddr );
      Rc = recvfrom( Ctx->Sock, Buf, BufSz, 0, &Ctx->FromAddr, &Rcvd );
      if( Rc == 0 || Rc == SOCKET_ERROR ) return -1;
      return Rc;
    }
    return 1;
  }

static int COMMON_SendData( NETCTX* Ctx, char* Buf, int BufSz )
  {
    if( Ctx->Stream ) {
      if( xsend( Ctx->Sock, Buf, BufSz, 0 ) != BufSz ) return 0;
    }
    else {
      if( sendto( Ctx->Sock, Buf, BufSz, 0,
                  &Ctx->Peer, sizeof( Ctx->Peer ) ) != BufSz ) return 0;
    }
    return 1;
  }

//------------- Common routines end ---------------------------------




#ifdef SUPPORT_SPX //---------- SPX implementation begin -------------------

static void SPX_SetLocalPort( NETCTX* Ctx, char* Port )
  {
    unsigned short p;

    Ctx->Local.sa_family = AF_IPX;
    RtlZeroMemory( Ctx->Local.sa_data, sizeof( Ctx->Local.sa_data ) );
    if( Port == NULL ) p = 0;
    else p = (unsigned short) Strtoul( Port, NULL, 0 );
    ((LPSOCKADDR_IPX) &Ctx->Local)->sa_socket = xhtons( p );
  }

static void SPX_GetLocalAddress( NETCTX* Ctx, char* AddrBuf, char* Port )
  {
    int            i;
    char           HDig[ 16 ];
    unsigned       Net;
    SOCKADDR_IPX   Addr;

    i = sizeof(Addr);
    if( xgetsockname( Ctx->Sock, (LPSOCKADDR) &Addr, &i ) != 0 ) return;
    Net = ((((unsigned) Addr.sa_netnum[0]) << 24) & 255) +
          ((((unsigned) Addr.sa_netnum[1]) << 16) & 255) +
          ((((unsigned) Addr.sa_netnum[2]) << 8) & 255) +
          (((unsigned) Addr.sa_netnum[3]) & 255);
    wsprintf( AddrBuf, "%08X.", Net );
    for( i = 0; i < 6; i++ ) {
      wsprintf( HDig, "%02X", Addr.sa_nodenum[ i ] );
      lstrcat( AddrBuf, HDig );
    }
    wsprintf( Port, "0x%04X", xntohs( Addr.sa_socket ) );
  }

static void SPX_GetDgramSrcAddress( NETCTX* Ctx, char* AddrBuf, char* Port )
  {
    int            i;
    char           HDig[ 16 ];
    unsigned       Net;
    SOCKADDR_IPX*  Addr;

    Addr = (LPSOCKADDR_IPX) &Ctx->FromAddr;
    Net = ((((unsigned) (Addr->sa_netnum[0])) << 24) & 255) +
          ((((unsigned) (Addr->sa_netnum[1])) << 16) & 255) +
          ((((unsigned) (Addr->sa_netnum[2])) << 8) & 255) +
          (((unsigned) (Addr->sa_netnum[3])) & 255);
    wsprintf( AddrBuf, "%08X.", Net );
    for( i = 0; i < 6; i++ ) {
      wsprintf( HDig, "%02X", Addr->sa_nodenum[ i ] );
      lstrcat( AddrBuf, HDig );
    }
    wsprintf( Port, "0x%04X", xntohs( Addr->sa_socket ) );
  }

static int SPX_SetPeerAddress( NETCTX* Ctx, char* Addr, char* Port )
  {
    char ABuf[ 64 ];
    char *cp, *cpx;
    int  i;
    unsigned long  p;
    unsigned char  a, b;

    Ctx->Peer.sa_family = AF_IPX;
    lstrcpy( ABuf, Addr ); CharUpper( ABuf );
    cp = Strchr( ABuf, '.' );
    if( cp == NULL ) { p = 0; cp = ABuf; }
    else p = Strtoul( ABuf, &cp, 16 );
    ((LPSOCKADDR_IPX) &Ctx->Peer)->sa_netnum[0] = (unsigned char)(p >> 24);
    ((LPSOCKADDR_IPX) &Ctx->Peer)->sa_netnum[1] = (unsigned char)(p >> 16);
    ((LPSOCKADDR_IPX) &Ctx->Peer)->sa_netnum[2] = (unsigned char)(p >> 8);
    ((LPSOCKADDR_IPX) &Ctx->Peer)->sa_netnum[3] = (unsigned char)p;
    while( ! ishexdigit( *cp ) ) cp++;
    cpx = cp;
    while( ishexdigit( *cpx ) ) cpx++;
    i = 5;
    do {
      a = b = 0;
      cpx--;
      if( *cpx >= '0' && *cpx <= '9' ) a = *cpx - '0';
      else if( *cpx >= 'A' && *cpx <= 'F' ) a = *cpx - 'A' + 10;
      else return 0;
      if( cpx != cp ) {
        cpx--;
        if( *cpx >= '0' && *cpx <= '9' ) b = *cpx - '0';
        else if( *cpx >= 'A' && *cpx <= 'F' ) b = *cpx - 'A' + 10;
        else return 0;
      }
      ((LPSOCKADDR_IPX) &Ctx->Peer)->sa_nodenum[ i-- ] = a + (b << 4);
#     ifdef DEBUG
        printf( "%02X", a + (b << 4) );
#     endif
    } while( cpx != cp && i >= 0 );
    while( i >= 0 ) ((LPSOCKADDR_IPX) &Ctx->Peer)->sa_nodenum[ i-- ] = 0;
    if( Port == NULL )
      p = Strtoul( GetDefaultPort( GetProtocol( Ctx ) ), NULL, 0 );
    else {
      p = Strtoul( Port, NULL, 0 );
      if( p == 0 )
        p = Strtoul( GetDefaultPort( GetProtocol( Ctx ) ), NULL, 0 );
    }
    ((LPSOCKADDR_IPX) &Ctx->Peer)->sa_socket = xhtons( (unsigned short) p );
#   ifdef DEBUG
      printf( ":%04X\n", p );
#   endif
    return 1;
  }

static int SPX_OpenPort( NETCTX* Ctx )
  {
    BOOL  BVal;
    struct linger lng;

    Ctx->Stream = 1;
    Ctx->Sock = xsocket( AF_IPX, SOCK_STREAM, NSPROTO_SPX );
    if( Ctx->Sock == INVALID_SOCKET ) return 0;
    if( xbind( Ctx->Sock, &Ctx->Local, sizeof( Ctx->Local ) ) == 0 ) {
      BVal = TRUE;
      xsetsockopt( Ctx->Sock, SOL_SOCKET,
                  SO_BROADCAST, (char*) &BVal, sizeof( BVal ) );
      lng.l_onoff = 1;
      lng.l_linger = 180;
      xsetsockopt( Ctx->Sock, SOL_SOCKET,
                  SO_LINGER, (char*) &lng, sizeof( lng ) );
      return 1;
    }
    xclosesocket( Ctx->Sock );
    Ctx->Sock = INVALID_SOCKET;
    return 0;
  }

static int SPX_OpenDgramPort( NETCTX* Ctx )
  {
    BOOL  BVal;

    Ctx->Stream = 0;
    Ctx->Sock = xsocket( AF_IPX, SOCK_DGRAM, NSPROTO_IPX + 4 );
    if( Ctx->Sock == INVALID_SOCKET ) return 0;
    if( xbind( Ctx->Sock, &Ctx->Local, sizeof( Ctx->Local ) ) == 0 ) {
      BVal = TRUE;
      xsetsockopt( Ctx->Sock, SOL_SOCKET,
                  SO_BROADCAST, (char*) &BVal, sizeof( BVal ) );
      return 1;
    }
    xclosesocket( Ctx->Sock );
    Ctx->Sock = INVALID_SOCKET;
    return 0;
  }

static void SPX_Deallocate( NETCTX* Ctx )
  {
    Ctx = NULL;
  }

static int SPX_Init( NETCTX* Ctx )
  {
    Ctx->SetLocalPort = (PSetLocalPort) SPX_SetLocalPort;
    Ctx->GetLocalAddress = (PGetLocalAddress) SPX_GetLocalAddress;
    Ctx->GetDgramSrcAddress = (PGetDgramSrcAddress) SPX_GetDgramSrcAddress;
    Ctx->SetPeerAddress = (PSetPeerAddress) SPX_SetPeerAddress;
    Ctx->OpenPort = (POpenPort) SPX_OpenPort;
    Ctx->OpenDgramPort = (POpenDgramPort) SPX_OpenDgramPort;
    Ctx->ClosePort = (PClosePort) COMMON_ClosePort;
    Ctx->ListenForConnection = (PListenForConnection) COMMON_ListenForConnection;
    Ctx->AcceptConnection = (PAcceptConnection) COMMON_AcceptConnection;
    Ctx->EstablishConnection = (PEstablishConnection) COMMON_EstablishConnection;
    Ctx->RecvData = (PRecvData) COMMON_RecvData;
    Ctx->SendData = (PSendData) COMMON_SendData;
    Ctx->Deallocate = (PDeallocate) SPX_Deallocate;
    return 1;
  }

#endif // SUPPORT_SPX ---------- SPX implementation end --------------------




#ifdef SUPPORT_TCP //---------- TCP implementation begin -------------------

static void TCP_SetLocalPort( NETCTX* Ctx, char* Port )
  {
    unsigned short  p;

    Ctx->Local.sa_family = AF_INET;
    RtlZeroMemory( Ctx->Local.sa_data, sizeof( Ctx->Local.sa_data ) );
    if( Port == NULL ) p = 0;
    else p = (unsigned short) Strtoul( Port, NULL, 0 );
    ((LPSOCKADDR_IN) &Ctx->Local)->sin_port = xhtons( p );
  }

static void TCP_GetLocalAddress( NETCTX* Ctx, char* AddrBuf, char* Port )
  {
    int           Rc, Sz;
    SOCKADDR_IN   Addr;

    Sz = sizeof(Addr);
    Rc = xgetsockname( Ctx->Sock, (LPSOCKADDR) &Addr, &Sz );
    if( Rc != 0 ) return;
    Rc = xntohl( Addr.sin_addr.s_addr );
    wsprintf( AddrBuf, "%d.%d.%d.%d", (Rc >> 24) & 255, (Rc >> 16) & 255,
              (Rc >> 8) & 255, Rc & 255 );
    wsprintf( Port, "%d", xntohs( Addr.sin_port ) );
  }

static void TCP_GetDgramSrcAddress( NETCTX* Ctx, char* AddrBuf, char* Port )
  {
    int           Rc;
    SOCKADDR_IN*  Addr;

    Addr = (SOCKADDR_IN*) &Ctx->FromAddr;
    Rc = xntohl( Addr->sin_addr.s_addr );
    wsprintf( AddrBuf, "%d.%d.%d.%d", (Rc >> 24) & 255, (Rc >> 16) & 255,
              (Rc >> 8) & 255, Rc & 255 );
    wsprintf( Port, "%d", xntohs( Addr->sin_port ) );
  }

static int TCP_SetPeerAddress( NETCTX* Ctx, char* Addr, char* Port )
  {
    char   *cp;
    ULONG  IPaddr, a, b;
    unsigned short p;

    Ctx->Peer.sa_family = AF_INET;
    IPaddr = 0;
    a = Strtoul( Addr, &cp, 10 );
    if( a != 0 && a < 256 && *cp == '.' )
      if( cp[1] != '\0' ) {
        b = a << 24;
        a = Strtoul( cp + 1, &cp, 10 );
        if( a < 256 && *cp == '.' )
          if( cp[1] != '\0' ) {
            b += a << 16;
            a = Strtoul( cp + 1, &cp, 10 );
            if( a < 256 && *cp == '.' )
              if( cp[1] != '\0' ) {
                b += a << 8;
                a = Strtoul( cp + 1, &cp, 10 );
                if( a < 256 && (*cp == '\0' || *cp == ':') ) IPaddr = b + a;
              }
          }
      }
    RtlZeroMemory( Ctx->Peer.sa_data, sizeof( Ctx->Peer.sa_data ) );
    ((LPSOCKADDR_IN) &Ctx->Peer)->sin_addr.s_addr = xhtonl( IPaddr );
    if( Port == NULL )
      p = (unsigned short)
              Strtoul( GetDefaultPort( GetProtocol( Ctx ) ), NULL, 0 );
    else {
      p = (unsigned short) Strtoul( Port, NULL, 0 );
      if( p == 0 )
        p = (unsigned short)
                Strtoul( GetDefaultPort( GetProtocol( Ctx ) ), NULL, 0 );
    }
    ((LPSOCKADDR_IN) &Ctx->Peer)->sin_port = xhtons( p );
    return 1;
  }

static int TCP_OpenPort( NETCTX* Ctx )
  {
    BOOL  BVal;
    //int   Opt;
    struct linger lng;

    Ctx->Stream = 1;
    Ctx->Sock = xsocket( AF_INET, SOCK_STREAM, 0 );
    if( Ctx->Sock == INVALID_SOCKET ) return 0;
    if( xbind( Ctx->Sock, &Ctx->Local, sizeof( Ctx->Local ) ) == 0 ) {
      BVal = TRUE;
      xsetsockopt( Ctx->Sock, SOL_SOCKET,
                  SO_BROADCAST, (char*) &BVal, sizeof( BVal ) );
      lng.l_onoff = 1;
      lng.l_linger = 180;
      xsetsockopt( Ctx->Sock, SOL_SOCKET,
                  SO_LINGER, (char*) &lng, sizeof( lng ) );

      // ??? winnt with standard ras has a problem with mtu over slip conn-s;
      // ??? disable nagle algorithm and set output buffer to 512 bytes
      // ??? (most inet providers use mtu=576)

      //Opt = 512;
      //xsetsockopt( Sock, SOL_SOCKET, SO_SNDBUF, (char*) &Opt, sizeof(Opt) );
      //BVal = TRUE;
      //xsetsockopt( Sock, IPPROTO_TCP, TCP_NODELAY,
      //             (char*) &BVal, sizeof( BVal ) );
      return 1;
    }
    xclosesocket( Ctx->Sock );
    Ctx->Sock = INVALID_SOCKET;
    return 0;
  }

static int TCP_OpenDgramPort( NETCTX* Ctx )
  {
    BOOL  BVal;

    Ctx->Stream = 0;
    Ctx->Sock = xsocket( AF_INET, SOCK_DGRAM, 0 );
    if( Ctx->Sock == INVALID_SOCKET ) return 0;
    if( xbind( Ctx->Sock, &Ctx->Local, sizeof( Ctx->Local ) ) == 0 ) {
      BVal = TRUE;
      xsetsockopt( Ctx->Sock, SOL_SOCKET,
                  SO_BROADCAST, (char*) &BVal, sizeof( BVal ) );
      return 1;
    }
    xclosesocket( Ctx->Sock );
    Ctx->Sock = INVALID_SOCKET;
    return 0;
  }

static void TCP_Deallocate( NETCTX* Ctx )
  {
    Ctx = NULL;
  }

static int TCP_Init( NETCTX* Ctx )
  {
    Ctx->SetLocalPort = (PSetLocalPort) TCP_SetLocalPort;
    Ctx->GetLocalAddress = (PGetLocalAddress) TCP_GetLocalAddress;
    Ctx->GetDgramSrcAddress = (PGetDgramSrcAddress) TCP_GetDgramSrcAddress;
    Ctx->SetPeerAddress = (PSetPeerAddress) TCP_SetPeerAddress;
    Ctx->OpenPort = (POpenPort) TCP_OpenPort;
    Ctx->OpenDgramPort = (POpenDgramPort) TCP_OpenDgramPort;
    Ctx->ClosePort = (PClosePort) COMMON_ClosePort;
    Ctx->ListenForConnection = (PListenForConnection) COMMON_ListenForConnection;
    Ctx->AcceptConnection = (PAcceptConnection) COMMON_AcceptConnection;
    Ctx->EstablishConnection = (PEstablishConnection) COMMON_EstablishConnection;
    Ctx->RecvData = (PRecvData) COMMON_RecvData;
    Ctx->SendData = (PSendData) COMMON_SendData;
    Ctx->Deallocate = (PDeallocate) TCP_Deallocate;
    return 1;
  }

#endif // SUPPORT_TCP ---------- TCP implementation end --------------------
